import dotenv from 'dotenv';
dotenv.config();

import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import rateLimit from 'express-rate-limit';
import jwt from 'jsonwebtoken';
import { body, validationResult } from 'express-validator';
import { makeWASocket, DisconnectReason, useMultiFileAuthState } from '@whiskeysockets/baileys';
import QRCode from 'qrcode';
import fs from 'fs';

const app = express();

// Security middleware
app.use(helmet());
app.use(cors({
    origin: process.env.FRONTEND_URL.split(','),
    credentials: true,
    methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
    allowedHeaders: ['Content-Type', 'Authorization']
}));
app.use(express.json({ limit: '10mb' }));

// Rate limiting
const limiter = rateLimit({
    windowMs: 15 * 60 * 1000,
    max: 100,
    message: { error: 'Too many requests' }
});

const messageLimiter = rateLimit({
    windowMs: 60 * 1000,
    max: 10,
    message: { error: 'Message rate limit exceeded' }
});

app.use(limiter);

// Global state
let sock = null;
let qrCode = null;
let isConnected = false;
const authDir = 'auth_info';

// Auth middleware
const authenticateToken = (req, res, next) => {
    const token = req.headers['authorization']?.split(' ')[1];
    if (!token) return res.status(401).json({ error: 'Access token required' });

    jwt.verify(token, process.env.JWT_SECRET, (err, user) => {
        if (err) return res.status(403).json({ error: 'Invalid token' });
        req.user = user;
        next();
    });
};

// Logger
const log = (status, message) => console.log(`[${new Date().toISOString()}] ${status}: ${message}`);

// WhatsApp connection
const connectToWhatsApp = async () => {
    try {
        log('INFO', 'Initializing WhatsApp connection...');
        const { state, saveCreds } = await useMultiFileAuthState(authDir);

        sock = makeWASocket({
            auth: state,
            printQRInTerminal: false,
            browser: ['WhatsApp API', 'Chrome', '1.0.0']
        });

        sock.ev.on('connection.update', async (update) => {
            const { connection, lastDisconnect, qr } = update;

            if (qr) {
                qrCode = await QRCode.toDataURL(qr);
                log('QR', 'QR code generated');
            }

            if (connection === 'close') {
                const shouldReconnect = lastDisconnect?.error?.output?.statusCode !== DisconnectReason.loggedOut;
                log('DISCONNECT', `Connection closed - Will reconnect: ${shouldReconnect}`);

                if (shouldReconnect) {
                    qrCode = null;
                    isConnected = false;
                    setTimeout(connectToWhatsApp, 5000);
                } else {
                    isConnected = false;
                    qrCode = null;
                }
            } else if (connection === 'open') {
                log('SUCCESS', 'WhatsApp connected');
                isConnected = true;
                qrCode = null;
            }
        });

        sock.ev.on('creds.update', saveCreds);
    } catch (error) {
        log('ERROR', `Connection failed: ${error.message}`);
        isConnected = false;
        qrCode = null;
        sock = null;
        setTimeout(connectToWhatsApp, 5000);
    }
};

// Format phone number
const formatNumber = (number) => {
    let clean = String(number).replace(/[^0-9]/g, '');
    if (clean.startsWith('0')) clean = '62' + clean.substring(1);
    if (!clean.startsWith('62')) clean = '62' + clean;
    return `${clean}@s.whatsapp.net`;
};

// Routes
app.post('/api/login', [
    body('username').notEmpty(),
    body('password').notEmpty()
], (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) return res.status(400).json({ errors: errors.array() });

    const { username, password } = req.body;
    if (username === process.env.ADMIN_USERNAME && password === process.env.ADMIN_PASSWORD) {
        const token = jwt.sign({ username, role: 'admin' }, process.env.JWT_SECRET, { expiresIn: '24h' });
        res.json({ token });
    } else {
        res.status(401).json({ error: 'Invalid credentials' });
    }
});

app.get('/api/status', authenticateToken, (req, res) => {
    res.json({ connected: isConnected, qrCode });
});

app.post('/api/send-message', messageLimiter, authenticateToken, [
    body('number').notEmpty(),
    body('message').notEmpty().isLength({ min: 1, max: 1000 })
], async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) return res.status(400).json({ errors: errors.array() });

    try {
        const { number, message } = req.body;
        if (!isConnected || !sock) return res.status(400).json({ error: 'WhatsApp not connected' });

        const jid = formatNumber(number);

        try {
            const [result] = await sock.onWhatsApp(jid);
            if (!result?.exists) return res.status(400).json({ error: 'Nomernya gak terdaftar nih bro' });
        } catch {
            return res.status(400).json({ error: 'Nomernya gak terdaftar nih bro' });
        }

        await sock.sendMessage(jid, { text: message });
        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

const clearSession = () => {
    if (fs.existsSync(authDir)) fs.rmSync(authDir, { recursive: true, force: true });
};

app.get('/api/logout', authenticateToken, async (req, res) => {
    try {
        if (sock) await sock.logout();

        clearSession();

        isConnected = false;
        qrCode = null;
        sock = null;
        res.json({ success: true });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.get('/api/reconnect', authenticateToken, async (req, res) => {
    try {
        if (sock) sock.end();

        clearSession();

        isConnected = false;
        qrCode = null;
        sock = null;

        setTimeout(connectToWhatsApp, 2000);
        res.json({ success: true, message: 'Reconnection started' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

app.get('/api/refresh-qr', authenticateToken, async (req, res) => {
    try {
        log('INFO', 'QR refresh requested');

        // No active socket or connection
        if (!sock) {
            return res.json({ success: false, message: 'No active connection' });
        }

        // Already connected, no refresh needed
        if (isConnected) {
            return res.json({ success: false, message: 'Already connected' });
        }

        // Force QR regeneration
        sock.end();
        sock = null;
        qrCode = null;

        setTimeout(connectToWhatsApp, 1000);

        return res.json({ success: true, message: 'QR refresh started' });
    } catch (error) {
        logConnection('ERROR', `QR refresh failed: ${error.message}`);
        return res.status(500).json({ error: error.message });
    }
});

app.get('/health', (req, res) => {
    res.json({
        status: 'OK',
        timestamp: new Date().toISOString(),
        whatsapp: isConnected
    });
});

// Initialize connection
connectToWhatsApp();

if (process.env.NODE_ENV !== 'production') {
    const PORT = process.env.PORT || 3001;
    app.listen(PORT, () => {
        console.log(`[${new Date().toISOString()}] INFO: API server started on port ${PORT}`);
    });
}

export default app;